"use client";
import { useState } from "react";
import { Loader2, AlertTriangle } from "lucide-react";
import { useLanguage } from "@/contexts/LanguageContext";
import {
  Dialog,
  DialogContent,
  DialogHeader,
  DialogTitle,
} from "@/components/ui/dialog";
import { Button } from "@/components/ui/button";

interface DeleteTechnicianDialogProps {
  isOpen: boolean;
  technicianName?: string | null;
  technicianEmail: string;
  onClose: () => void;
  onConfirm: (unassignFromTasks?: boolean, unassignFromTickets?: boolean) => Promise<void>;
}

export default function DeleteTechnicianDialog({
  isOpen,
  technicianName,
  technicianEmail,
  onClose,
  onConfirm,
}: DeleteTechnicianDialogProps) {
  const { t } = useLanguage();
  const [deleting, setDeleting] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [canUnassign, setCanUnassign] = useState(false);
  const [assignedTasksCount, setAssignedTasksCount] = useState(0);
  const [assignedTasks, setAssignedTasks] = useState<Array<{ id: string; title: string }>>([]);
  const [unassignFromTasks, setUnassignFromTasks] = useState(false);
  const [canUnassignTickets, setCanUnassignTickets] = useState(false);
  const [assignedTicketsCount, setAssignedTicketsCount] = useState(0);
  const [assignedTickets, setAssignedTickets] = useState<Array<{ id: string; title: string }>>([]);
  const [unassignFromTickets, setUnassignFromTickets] = useState(false);

  const handleConfirm = async () => {
    setDeleting(true);
    setError(null);
    try {
      await onConfirm(unassignFromTasks, unassignFromTickets);
      onClose();
    } catch (error: any) {
      console.error("Failed to delete technician:", error);
      const errorMessage = error?.message || "Failed to delete technician";
      setError(errorMessage);
      
      // Check if error has canUnassign flag from API for tasks
      if (error?.canUnassign) {
        setCanUnassign(true);
        setAssignedTasksCount(error.assignedTasksCount || 0);
        setAssignedTasks(error.assignedTasks || []);
      } else if (errorMessage.includes("assigned to") && errorMessage.includes("task(s)")) {
        // Fallback: Extract task count from error message
        const match = errorMessage.match(/assigned to (\d+) task\(s\)/);
        if (match) {
          setCanUnassign(true);
          setAssignedTasksCount(parseInt(match[1], 10));
        }
      }

      // Check if error has canUnassignTickets flag from API
      if (error?.canUnassignTickets) {
        setCanUnassignTickets(true);
        setAssignedTicketsCount(error.assignedTicketsCount || 0);
        setAssignedTickets(error.assignedTickets || []);
      } else if (errorMessage.includes("assigned to") && errorMessage.includes("ticket(s)")) {
        // Fallback: Extract ticket count from error message
        const match = errorMessage.match(/assigned to (\d+) ticket\(s\)/);
        if (match) {
          setCanUnassignTickets(true);
          setAssignedTicketsCount(parseInt(match[1], 10));
        }
      }
    } finally {
      setDeleting(false);
    }
  };

  const handleClose = () => {
    setError(null);
    setCanUnassign(false);
    setAssignedTasksCount(0);
    setAssignedTasks([]);
    setUnassignFromTasks(false);
    setCanUnassignTickets(false);
    setAssignedTicketsCount(0);
    setAssignedTickets([]);
    setUnassignFromTickets(false);
    onClose();
  };

  if (!isOpen) return null;

  return (
    <Dialog
      open={isOpen}
      onOpenChange={(open) => !open && handleClose()}
      modal={true}
    >
      <DialogContent className='sm:max-w-[500px]'>
        <DialogHeader>
          <DialogTitle className='flex items-center gap-2 text-red-600'>
            <AlertTriangle className='w-5 h-5' />
            {t("admin.confirmDeletion")}
          </DialogTitle>
        </DialogHeader>

        <div className='py-4'>
          <div className='rounded-lg bg-red-50 p-4 border border-red-200 mb-4'>
            <div className='flex items-start gap-3'>
              <AlertTriangle className='w-5 h-5 text-red-600 mt-0.5' />
              <div className='flex-1'>
                <p className='text-sm font-medium text-red-900 mb-2'>
                  {t("admin.cannotBeUndone")}
                </p>
                <p className='text-sm text-red-700'>
                  {t("admin.deleteTechnicianConfirm")}
                </p>
              </div>
            </div>
          </div>

          <div className='bg-gray-50 p-4 rounded-lg border border-gray-200'>
            <p className='text-sm font-medium text-gray-900 mb-2'>
              {t("admin.technicians")}:
            </p>
            <p className='text-sm text-gray-700'>
              {technicianName || "—"} ({technicianEmail})
            </p>
          </div>

          {error && (
            <div className='mt-4 p-3 bg-red-50 border border-red-200 rounded-lg'>
              <p className='text-sm text-red-800 mb-3'>{error}</p>
              
              {/* Show assigned tasks list */}
              {canUnassign && assignedTasks.length > 0 && (
                <div className='mt-3 pt-3 border-t border-red-200'>
                  <p className='text-xs font-medium text-red-900 mb-2'>
                    Assigned Tasks ({assignedTasksCount}):
                  </p>
                  <ul className='text-xs text-red-700 space-y-1 mb-3 max-h-32 overflow-y-auto'>
                    {assignedTasks.map((task) => (
                      <li key={task.id} className='pl-2'>• {task.title}</li>
                    ))}
                  </ul>
                  <label className='flex items-start gap-2 cursor-pointer'>
                    <input
                      type='checkbox'
                      checked={unassignFromTasks}
                      onChange={(e) => setUnassignFromTasks(e.target.checked)}
                      className='mt-1 w-4 h-4 text-red-600 border-red-300 rounded focus:ring-red-500'
                    />
                    <div className='flex-1'>
                      <p className='text-sm font-medium text-red-900'>
                        Remove from all {assignedTasksCount} task(s) and delete
                      </p>
                      <p className='text-xs text-red-700 mt-1'>
                        This will unassign the technician from all tasks before deletion.
                      </p>
                    </div>
                  </label>
                </div>
              )}

              {/* Show assigned tickets list */}
              {canUnassignTickets && assignedTickets.length > 0 && (
                <div className='mt-3 pt-3 border-t border-red-200'>
                  <p className='text-xs font-medium text-red-900 mb-2'>
                    Assigned Tickets ({assignedTicketsCount}):
                  </p>
                  <ul className='text-xs text-red-700 space-y-1 mb-3 max-h-32 overflow-y-auto'>
                    {assignedTickets.map((ticket) => (
                      <li key={ticket.id} className='pl-2'>• {ticket.title}</li>
                    ))}
                  </ul>
                  <label className='flex items-start gap-2 cursor-pointer'>
                    <input
                      type='checkbox'
                      checked={unassignFromTickets}
                      onChange={(e) => setUnassignFromTickets(e.target.checked)}
                      className='mt-1 w-4 h-4 text-red-600 border-red-300 rounded focus:ring-red-500'
                    />
                    <div className='flex-1'>
                      <p className='text-sm font-medium text-red-900'>
                        Remove from all {assignedTicketsCount} ticket(s) and delete
                      </p>
                      <p className='text-xs text-red-700 mt-1'>
                        This will unassign the technician from all tickets before deletion.
                      </p>
                    </div>
                  </label>
                </div>
              )}

              {/* Show warning about audit trail if completions exist */}
              {error.includes("completion") && (
                <div className='mt-3 pt-3 border-t border-red-200'>
                  <p className='text-xs text-red-700 italic'>
                    Note: Deletion is blocked to preserve audit trail and prevent denial of completed work.
                  </p>
                </div>
              )}
            </div>
          )}
        </div>

        <div className='flex justify-end gap-3 pt-4 border-t'>
          <Button variant='outline' onClick={handleClose} disabled={deleting}>
            {t("common.cancel") || "Cancel"}
          </Button>
          <Button
            variant='destructive'
            onClick={handleConfirm}
            disabled={deleting}
            className='bg-red-600 hover:bg-red-700 text-white'
          >
            {deleting ? (
              <>
                <Loader2 className='mr-2 h-4 w-4 animate-spin' />
                {t("admin.deleting") || "Deleting..."}
              </>
            ) : (
              t("admin.deleteTechnician") || "Delete Technician"
            )}
          </Button>
        </div>
      </DialogContent>
    </Dialog>
  );
}
